#ifndef RAY318_INCLUDED // -*- C++ -*-
#define RAY318_INCLUDED

/*************************************************************************

  Standard header for the 318 ray tracer.

  $Id: ray318.h,v 1.24 2001/04/09 19:40:41 garland Exp $

 *************************************************************************/

#include <gfx/gfx.h>
#include <gfx/mat4.h>
#include <gfx/raster.h>

#include <vector>
#include <string>

using namespace std;


////////////////////////////////////////////////////////////////////////
//
// Standard types for use in 318 ray tracer
//

// While both are represented as RGB triples, we try to make a
// distinction between colors and reflectances.  Colors represent
// light being transported from sources to the eye.  Reflectances
// represent the fraction of incoming light reflected by a surface.
//
typedef Vec3 rgbRefl;
typedef Vec3 rgbColor;

// This camera class mirrors the parameterization of the OpenGL camera model
// (see gluLookAt() for details).
//
class Camera
{
public:
    Vec3 eye, at, up;		// Configuration of camera
    double fovy, aspect;	// Field-of-view & aspect ratio of image
    int width, height;		// Pixel dimensions of output image


    // The camera configuration and perspective projection specified in
    // the scene file determine the transformation from world to screen
    // coordinates.
    //
    // NOTE: These are INVALID until compute_transform() is called.
    //
    Mat4 M_ws;			// world->screen transformation
    Mat4 M_sw;			// screen->world transformation

    Mat4& compute_transform();
};

// Only point lights are supported
//
class Light
{
public:
    Vec3 position;		// Position of the point light
    rgbColor rgb;		// Color of light it is emitting
};

// Standard material components
//
class Material
{
public:
    Material();

    rgbColor emit;		// Light being emitted

    rgbRefl r_amb;		// Ambient reflectance
    rgbRefl r_diff;		// Diffuse reflectance
    rgbRefl r_spec;		// Specular reflectance (for reflection)
    rgbRefl r_trans;		// Specular transmittance (for transmission)

    double shininess;		// The Phong specular exponent

    double ir;			// Index of refraction

	ByteRaster *texture;
};

////////////////////////////////////////////////////////////////////////
//
// Scene file format support
//
//   The scene management code assumes that all geometric primitives
//   are derived from the following minimal rayObject class.
//
//   NOTE: These classes are just skeletal definitions.  You will almost
//         certainly want to modify/extend them.
//

class Ray
{
public:
    Vec3 p;		// origin of ray
    Vec3 d;		// (unit) direction vector of ray
};


class rayObject
{
public:

	virtual int ComputeIntersection(double *t, Ray *ray) = 0;
	virtual Vec3 ComputeNormal(Vec3 *intersect) = 0;

    Material *material;
};

class  raySphere : public rayObject
{
private:
    Vec3 center;
    double radius;

public:

    raySphere(double r, const Vec3& ctr) { center=ctr; radius=r; }

	inline int ComputeIntersection(double *t, Ray *ray)
	{
		double a = 0, b = 0, c = 0;
		double d = 0, t1 = 0, t2 = 0;

		a = 1.0;		//(ray->d * ray->d);
		b = 2 * ((ray->p * ray->d) - (ray->d * center));
		c = (ray->p * ray->p) - (ray->p * center) * 2 + (center * center) - (radius * radius);

		d = (b*b) - (4 * /* a * */ c);
		if(d < 0)
		{
			return 0;
		}
		else
		{
			d = sqrt(d);
			t1 = (-b + d) / 2.0;	//(2 * a);
			t2 = (-b - d) / 2.0;	//(2 * a);

			if((t1 <= 0) && (t2 <= 0)) return 0;
			if(t1 <= 0) t1 = t2;
			if(t2 <= 0) t2 = t1;

			if(t1 < t2)
			{
				*t = t1;
			}
			else
			{
				*t = t2;
			}
			return 1;
		}
	}

	inline Vec3 ComputeNormal(Vec3 *intersect)
	{
		return ((*intersect) - center);
	}

};

class rayPolygon : public rayObject
{
private:
    vector<Vec3> vertices;

public:

    rayPolygon(int n, Vec3 *verts) : vertices(n)
    {
		for(int i=0; i<n; i++)  vertices[i] = verts[i];
    }

	int ComputeIntersection(double *t, Ray *ray)
	{
		return 0;
	}

	Vec3 ComputeNormal(Vec3 *intersect)
	{
		return Vec3(0, 0, 0);
	}
};

class rayQuadric : public rayObject
{
private:
    Mat4 mat;

public:

    rayQuadric(Mat4 m)
    {
		mat = m;
    }

	int ComputeIntersection(double *t, Ray *ray)
	{
		return 0;
	}

	Vec3 ComputeNormal(Vec3 *intersect)
	{

		return Vec3(0, 0, 0);
	}
};

class Scene
{
public:
    // All objects & lights read in from the scene file will be
    // collected in these lists.
    //
    std::vector<Light> lights;
    std::vector<rayObject *> objects;

    Camera camera;
    rgbColor background_color;
    rgbColor ambient_light;

public:
    //////////////////////////////////////////////////////////////////////
    //
    // Public interface used by scene parsing code to build the scene.
    //

    // Puts new object into the world, and also performs standard
    // tasks such as linking it to the current material.
    void add_object(rayObject *obj);

    // Any positions/directions associated with objects can be
    // transformed by the current transformation (read from scene
    // file) using these functions
    void transform(Vec3& v, double w);
    void transform_point(Vec3& p)     { transform(p, 1.0); }
    void transform_direction(Vec3& v) { transform(v, 0.0); }
};

extern Scene scene;


//
// Standard interface for creating materials.  This is primarily
// intended for the internal use of the scene parser.
//
extern Material *create_material(const string name);
extern Material *lookup_material(const string name);
extern Material *current_material();


// Global limit on the recursion depth of the raytracing process.
extern int ray_depth_limit;

extern void read_scene_file(const char *filename);

// RAY318_INCLUDED
#endif
