/*************************************************************************

  CS 318: MP4 -- Raytracer
    see http://www-courses.cs.uiuc.edu/~cs318/mp/mp4/mp4.html for details

  NAME   : Benjamin Gottemoller
  NET ID : gottemol0

 *************************************************************************/

#include "ray318.h"
#include <gfx/gui.h>
#include <gfx/raster.h>
#include <gfx/gl.h>
#include <gfx/script.h>

#include <FL/fl_file_chooser.H>

#define RAY_OFFSET		1E-8
#define MAX_FLOAT		1.7E308

#define CLAMP(value, a, b) { if(value < a) value = a; if(value > b) value = b; }
#define ABS(a)	((a < 0) ? (-(a)) : (a))


class GUI : public MxGUI
{
public:
    bool did_jitter;
    bool did_supersample;

    virtual void setup_for_drawing();
    virtual void draw_contents();
    virtual void cmdline_file(const char *file);
    virtual int cmdline_option(int argc, char **argv, int& index);
};

GUI gui;


int ray_depth_limit = 3;	 // probably don't need to change this
char *output_file = NULL;	 // you should not need to use this variable

bool will_supersample = false;	 // supersampling is off by default
bool will_jitter = false;	 // as is jittering
float supersample_factor = 3.0f; // default supersampling is 3x3 grid

//
// The image being rendered is stored here.  If we're in the process
// of rendering it, we draw it scanline by scanline.  If the entire
// image is complete, we draw the whole thing when repainting the window.
//
ByteRaster *img = NULL;
bool image_is_complete = false;

inline double Vec3Length(Vec3 &v);
inline void Vec3Normalize(Vec3 &v);
inline double Vec4Length(Vec4 &v);
inline void Vec4Normalize(Vec4 &v);

inline rgbColor ComputePhong(double I_l, double k_d, double k_s, Vec3 &n, Vec3 &L, Vec3 &r, Vec3 &V);
inline void ComputeEyeRay(Ray *ray, int x, int y);
inline int FindClosestRayHit(double *t, int *obj_index, Ray *r);
inline rgbColor TraceRay(Ray *r);
inline rgbColor Shade(int obj_index, Ray *ray, double t);

///////////////////////////////////////////////////////////////////////////////

inline double Vec3Length(Vec3 &v)
{
	return sqrt((v[0] * v[0]) + (v[1] * v[1]) + (v[2] * v[2])); 
}

inline void Vec3Normalize(Vec3 &v)
{
	v /= sqrt((v[0] * v[0]) + (v[1] * v[1]) + (v[2] * v[2])); 
}

inline double Vec4Length(Vec4 &v)
{
	return sqrt((v[0] * v[0]) + (v[1] * v[1]) + (v[2] * v[2]) + (v[3] * v[3])); 
}

inline void Vec4Normalize(Vec4 &v)
{
	v /= sqrt((v[0] * v[0]) + (v[1] * v[1]) + (v[2] * v[2]) + (v[3] * v[3])); 
}

inline rgbColor ComputePhong(rgbColor &I_l, rgbColor &k_d, rgbColor &k_s, double &shininess, Vec3 &n, Vec3 &L, Vec3 &r, Vec3 &V)
{
	rgbColor color(0, 0, 0);
	double n_dot_l = (n * L);
	double r_dot_v = (r * V);
	double specular = 0;

	if(n_dot_l > 0)
	{
		color[0] = I_l[0] * k_d[0] * n_dot_l;
		color[1] = I_l[1] * k_d[1] * n_dot_l;
		color[2] = I_l[2] * k_d[2] * n_dot_l;
	}

	if(r_dot_v > 0)
	{
		specular = pow(r_dot_v, shininess);
		color[0] += I_l[0] * k_s[0] * specular;
		color[1] += I_l[1] * k_s[1] * specular;
		color[2] += I_l[2] * k_s[2] * specular;
	}

	return color;
}

inline void ComputeEyeRay(Ray *ray, int x, int y)
{
	Vec4 tmp, q_s((double)x + 0.5, (double)y + 0.5, 0, 1), q_w;
	
	q_w = scene.camera.M_sw * q_s;
	ray->p = scene.camera.eye;
	ray->d = proj(q_w) - ray->p;
	Vec3Normalize(ray->d);
}

inline int FindClosestRayHit(double *t, int *obj_index, Ray *r)
{
	int i = 0, num_objects = scene.objects.size();
	double current_t = 0, closest_t = MAX_FLOAT;
	rayObject *obj = NULL;

	for(i=0; i<num_objects; i++)
	{
		obj = scene.objects[i];
		if(obj->ComputeIntersection(&current_t, r))
		{
			if(current_t < closest_t)
			{
				closest_t = current_t;
				if(obj_index != NULL) 
				{
					*obj_index = i;
				}
			}
		}
	}

	if(closest_t < MAX_FLOAT)
	{
		if(t != NULL)
		{
			*t = closest_t;
		}
		return 1;
	}

	return 0;
}

inline rgbColor TraceRay(Ray *r)
{	
	rgbColor color;
	double closest_t = MAX_FLOAT;
	int intersected_obj_index = 0;

	if(FindClosestRayHit(&closest_t, &intersected_obj_index, r))
	{ 
		color = Shade(intersected_obj_index, r, closest_t);
	}
	else
	{
		color = scene.background_color;
	}

	return color;
}

inline rgbColor Shade(int obj_index, Ray *ray, double t)
{
	static int recursion_level = 0;
	rayObject *obj = scene.objects[obj_index];
	int i=0;
	int result = 0;
	int num_lights = scene.lights.size();
	double closest_t = 0;
	double light_dist = 0;
	double diffuse = 0, specular = 0;
	Light *light = NULL;
	rgbColor color, refl_color, trans_color;
	Vec3 P, L, n, r, V;
	
	Ray shadow_ray, refl_ray, trans_ray;

	P = ray->p + (t * ray->d);
	n = obj->ComputeNormal(&P);
	Vec3Normalize(n);
	V = ray->d;

	color[0] = obj->material->r_amb[0] * scene.ambient_light[0];
	color[1] = obj->material->r_amb[1] * scene.ambient_light[1];
	color[2] = obj->material->r_amb[2] * scene.ambient_light[2];

	for(i=0; i<num_lights; i++)
	{
		light = &(scene.lights[i]);
		L = light->position - P;
		light_dist = Vec3Length(L);
		L /= light_dist;

		shadow_ray.p = P;
		shadow_ray.d = L;
		shadow_ray.p += shadow_ray.d * RAY_OFFSET;
		result = FindClosestRayHit(&closest_t, NULL, &shadow_ray);
		if((result == 0) || (closest_t >= light_dist))
		{
			r = (L - (2 * (n * L)) * n);
	
			color += ComputePhong(light->rgb, obj->material->r_diff, obj->material->r_spec, 
								  obj->material->shininess, n, L, r, V);
		}
	}

	if(recursion_level < ray_depth_limit)
	{
		recursion_level++;

		r = (V - (2 * (n * V)) * n);

		refl_ray.p = P;
		refl_ray.d = r;
		refl_ray.p += refl_ray.d * RAY_OFFSET;
		refl_color = TraceRay(&refl_ray);
		color[0] += (obj->material->r_spec[0] * refl_color[0]);
		color[1] += (obj->material->r_spec[1] * refl_color[1]);
		color[2] += (obj->material->r_spec[2] * refl_color[2]);

		recursion_level--;
	}

	return color;
}


void raytrace_image()
{
	Ray ray;
	rgbColor color;
	int r = 0, g = 0, b = 0;

    int w = img->width(),  h = img->height();
    memset(img->head(), 0, img->length());

    scene.camera.width = w;
    scene.camera.height = h;
    scene.camera.aspect = (double)w / (double)h;
    scene.camera.compute_transform();

    // iterate over all scanlines
    for(int j=0; j<h; j++)
    {
		// iterate over the pixels
		for(int i=0; i<w; i++)
		{
			ComputeEyeRay(&ray, i, j);
			color = TraceRay(&ray);

			CLAMP(color[0], 0, 1)
			CLAMP(color[1], 0, 1)
			CLAMP(color[2], 0, 1)

			r = (int)(color[0] * 255.0f + 0.5f);
			g = (int)(color[1] * 255.0f + 0.5f);
			b = (int)(color[2] * 255.0f + 0.5f);

			img->pixel(i, j)[0] = (UCHAR)r;
			img->pixel(i, j)[1] = (UCHAR)g;
			img->pixel(i, j)[2] = (UCHAR)b;
		}

		if( gui.canvas->visible() )
		{
			glRasterPos2f(0, h-j-1);
			glDrawPixels(w, 1, GL_RGB, GL_UNSIGNED_BYTE, img->pixel(0, j));
		}
    }
}

void setup_image(int w, int h)
{
    if(img)  delete img;
    img = new ByteRaster(w, h, 3);
    memset(img->head(), 0, img->length());
    image_is_complete = false;

}

////////////////////////////////////////////////////////////////////////
//
// The following functions manage the ray tracer GUI.  You shouldn't
// need to change them.
//

void GUI::setup_for_drawing()
{
    setup_image(gui.canvas->w(), gui.canvas->h());
    glClearColor(0.9f, 0.0f, 0.0f, 0.0f);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_LIGHTING);
    glDisable(GL_LIGHT0);
    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);


    // Prepare the OpenGL canvas for drawing.  We set things up to
    // directly draw into the pixels, since the ray engine will handle
    // all the appropriate transformations and projections.
    //
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluOrtho2D(0, gui.canvas->w(), 0, gui.canvas->h());

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
}

void GUI::draw_contents()
{
    int w = gui.canvas->w();
    int h = gui.canvas->h();


    glClear(GL_COLOR_BUFFER_BIT);

    if( did_jitter!=will_jitter || did_supersample!=will_supersample )
	image_is_complete = false;

    if( image_is_complete )
    {
		int w = img->width(),  h = img->height();
		int dh = canvas->h()-img->height();
		for(int j=0; j<h; j++)
		{
			glRasterPos2i(0, h-j-1 + dh);
			glDrawPixels(w, 1, GL_RGB, GL_UNSIGNED_BYTE, img->pixel(0, j));
		}
    }
    else
    {
		did_jitter = will_jitter;
		did_supersample = will_supersample;

		raytrace_image();
		image_is_complete = true;
    }
}

void GUI::cmdline_file(const char *file)
{
    if( file )
    {
		read_scene_file(file);
    }
    else
    {
		const char *filename =
	    fl_file_chooser("Select scene file:", "*.scn", "");

		if( !filename )  exit(0);

		read_scene_file(filename);
    }
}

int GUI::cmdline_option(int argc, char **argv, int& index)
{
    if( !strcmp("-j", argv[index]) )
	will_jitter = true;
    else if( !strcmp("-S", argv[index]) )
	will_supersample = true;

    else if( !strcmp("-D", argv[index]) )
	ray_depth_limit = atoi(argv[++index]);

    else if( !strcmp("-o", argv[index]) )
	output_file = argv[++index];

    else
	return 0;

    index++;
    return 1;
}

main(int argc, char **argv)
{
    gui.initialize(argc, argv, NULL, 320, 200);

    if( output_file )
    {
		setup_image(640, 480);
		raytrace_image();
		write_image(output_file, *img);
		exit(0);
    }

    gui.canvas->mode(FL_RGB8|FL_DEPTH|FL_SINGLE);

//    gui.add_toggle_menu("&Ray tracing/Will jitter", 0, will_jitter);
//    gui.add_toggle_menu("&Ray tracing/Will supersample", 0, will_supersample);

    return gui.run();
}
