/************************************************************************

  CS 318: MP3 -- The World in Motion
    see http://www-courses.cs.uiuc.edu/~cs318/mp/mp3.html for details

  NAME   : Benjamin Gottemoller
  NET ID : gottemol

 ************************************************************************/

#include "GLEngine.h"


COLOR::COLOR(float r, float g, float b, float a)
{
	COLOR::r = r;
	COLOR::g = g;
	COLOR::b = b;
	COLOR::a = a;
}

COLOR::COLOR(int r, int g, int b, int a)
{
	COLOR::r = ((float)r / 256.0f);
	COLOR::g = ((float)g / 256.0f);
	COLOR::b = ((float)b / 256.0f);
	COLOR::a = ((float)a / 256.0f);	
}

const COLOR& COLOR::operator=(COLOR &color)
{
	r = color.r;
	g = color.g;
	b = color.b;
	a = color.a;
	return *this;
}

COLOR COLOR::operator*(float value)
{
	return COLOR(r * value, g * value, b * value, a * value);
}

COLOR COLOR::operator/(float value)
{
	return COLOR(r / value, g / value, b / value, a / value);
}

GL_Material::GL_Material()
{
	memset(&Diffuse, 0, sizeof(COLOR));
	memset(&Ambient, 0, sizeof(COLOR));
	memset(&Emissive, 0, sizeof(COLOR));
	memset(&Specular, 0, sizeof(COLOR));
	Power = 0;
}

void GL_Material::SetMaterial()
{
	float value[4];

	const GLenum FACE = GL_FRONT;

	value[0] = Emissive.r;
	value[1] = Emissive.g;
	value[2] = Emissive.b;
	value[3] = Emissive.a;    
	glMaterialfv(FACE, GL_EMISSION, value);

	value[0] = Ambient.r;
	value[1] = Ambient.g;
	value[2] = Ambient.b;
	value[3] = Ambient.a;    
    glMaterialfv(FACE, GL_AMBIENT, value);

	value[0] = Diffuse.r;
	value[1] = Diffuse.g;
	value[2] = Diffuse.b;
	value[3] = Diffuse.a;    
    glMaterialfv(FACE, GL_DIFFUSE, value);

	value[0] = Specular.r;
	value[1] = Specular.g;
	value[2] = Specular.b;
	value[3] = Specular.a;    
    glMaterialfv(FACE, GL_SPECULAR, value);

    glMaterialf(FACE, GL_SHININESS, Power);
}

GL_Object::GL_Object(GL_Material *material, GL_FrameRateTracker *timer)
{ 
	Timer = timer;
	Yaw = Pitch = Roll = 0;
	Material.Diffuse = material->Diffuse;
	Material.Ambient = material->Ambient;
	Material.Emissive = material->Emissive;
	Material.Specular = material->Specular;
	Material.Power = material->Power;

	Scale = VECTOR3(1, 1, 1);
	IsAnimationPlaying = 0;
}

void GL_Object::Update()
{
	if(IsAnimationPlaying)
	{
		KFS.Update(Timer->TimeDiff);
		Position = KFS.Position;
		Yaw = KFS.Rotation.x;		
		Pitch = KFS.Rotation.y;		
		Roll = KFS.Rotation.z;
	}
}

void GL_Object::Render()
{
	MATRIX trans_mat;
    Material.SetMaterial();

	trans_mat.LoadTranslation(Position.x, Position.y, Position.z);
	Transform = Transform * trans_mat;
}

void GL_Object::RenderBoundingBox()
{
	glPushMatrix();

    VECTOR3 min, max;
    ComputeBoundingBox(min, max);
	
    glPushAttrib(GL_LIGHTING_BIT);
    glDisable(GL_LIGHTING);
	glDisable(GL_TEXTURE_2D);
	
    glColor3f(1.0f, 1.0f, 1.0f);
    
    glBegin(GL_LINE_LOOP);
    glVertex3d(min.x,min.y,min.z); glVertex3d(min.x,max.y,min.z);
    glVertex3d(max.x,max.y,min.z); glVertex3d(max.x,min.y,min.z);
    glEnd();

    glBegin(GL_LINE_LOOP);
    glVertex3d(min.x,min.y,max.z); glVertex3d(min.x,max.y,max.z);
    glVertex3d(max.x,max.y,max.z); glVertex3d(max.x,min.y,max.z);
    glEnd();

    glBegin(GL_LINES);
    glVertex3d(min.x,min.y,min.z); glVertex3d(min.x,min.y,max.z);
    glVertex3d(min.x,max.y,min.z); glVertex3d(min.x,max.y,max.z);
    glVertex3d(max.x,max.y,min.z); glVertex3d(max.x,max.y,max.z);
    glVertex3d(max.x,min.y,min.z); glVertex3d(max.x,min.y,max.z);
    glEnd();
    
	glEnable(GL_LIGHTING);
	glEnable(GL_TEXTURE_2D);

	glPopAttrib();
	glPopMatrix();
}


GL_FrameRateTracker *GL_HierarchicalModel::Timer;
int GL_HierarchicalModel::IsAnimationPlaying;
GL_HierarchicalModel::GL_HierarchicalModel()
{
	IsAnimationPlaying = 0;
	Yaw = Pitch = Roll = 0;
	Quadric = NULL;
	NumChildren = 0;
	Children = NULL;
}

void GL_HierarchicalModel::InitModel(int num_children, int quadric_type)
{
	QuadricType = quadric_type;
	if(!Quadric) 
	{
		Quadric = gluNewQuadric();
	}
	NumChildren = num_children;
	Children = new GL_HierarchicalModel[NumChildren];

	if((Children == NULL) || (Quadric == NULL))
	{
		fl_message("Error allocating hierarchical model memory. (Line: %d)", __LINE__);
	}
}

void GL_HierarchicalModel::LoadKeyFrames(char *directory_path, GL_HierarchicalModel *parent)
{
	char path[256];
	char file[256];

	if(parent == NULL)
	{
		sprintf(file, "%s/%s", directory_path, "root_key_frames.cfg");
		KFS.LoadKeys(file);
	}
	else
	{
		sprintf(file, "%s/%s", directory_path, "child_key_frames.cfg");
		KFS.LoadKeys(file);
	}

	for(int i=0; i<NumChildren; i++)
	{
		sprintf(path, "%s/child_%d", directory_path, i);
		Children[i].LoadKeyFrames(path, this);
	}
}

void GL_HierarchicalModel::SetupProperties(GL_Material *material, GL_FrameRateTracker *timer)
{
	Timer = timer;
	Material.Diffuse = material->Diffuse;
	Material.Ambient = material->Ambient;
	Material.Emissive = material->Emissive;
	Material.Specular = material->Specular;
	Material.Power = material->Power;
}

void GL_HierarchicalModel::Destroy()
{
	for(int i=0; i<NumChildren; i++)
	{
		Children[i].Destroy();
	}

	if(Children != NULL)
	{
		delete[] Children;
		Children = NULL;

		NumChildren = 0;
	}

	if(Quadric != NULL)
	{
		gluDeleteQuadric(Quadric);
		Quadric = NULL;
	}
}

void GL_HierarchicalModel::UpdateAndRenderHierarchy(GL_HierarchicalModel *parent)
{
	MATRIX Rotation, yaw_mat, pitch_mat, roll_mat;
	VECTOR3 Look_Vector(0,0,1);
	VECTOR3 Up_Vector(0,1,0);
	VECTOR3 Right_Vector(1,0,0);

	glPushMatrix();

	if(IsAnimationPlaying && KFS.IsReady())
	{
		KFS.Update(Timer->TimeDiff);
		if(parent == NULL)
		{
			Position = KFS.Position;
		}
		Yaw = KFS.Rotation.x;		
		Pitch = KFS.Rotation.y;		
		Roll = KFS.Rotation.z;
	}

	if(parent == NULL)
	{
		glTranslatef(Position.x, Position.y, Position.z);
	}

	if(parent != NULL)
	{
		glTranslatef(Joint.x, Joint.y, Joint.z);
	}
	glTranslatef(Center.x, Center.y, Center.z);

	yaw_mat = CreateAxisAngleRotationMat(Yaw, Up_Vector.x, Up_Vector.y, Up_Vector.z);
	Look_Vector = yaw_mat * Look_Vector;
	Right_Vector = yaw_mat * Right_Vector;	
	
	pitch_mat = CreateAxisAngleRotationMat(Pitch, Right_Vector.x, Right_Vector.y, Right_Vector.z);		
	Look_Vector = pitch_mat * Look_Vector;
	Up_Vector = pitch_mat * Up_Vector;
	
	roll_mat = CreateAxisAngleRotationMat(Roll, Look_Vector.x, Look_Vector.y, Look_Vector.z);		
	Up_Vector = roll_mat * Up_Vector;
	Right_Vector = roll_mat * Right_Vector;
	Rotation = yaw_mat * pitch_mat * roll_mat;

	Rotation.Transpose();
	glMultMatrixf(Rotation.matrix);
	
	glTranslatef(-Center.x, -Center.y, -Center.z);
	
	Material.SetMaterial();

    if(QuadricType == 1)
	{
		gluCylinder(Quadric, 0.1f, 0.1f, 1, 30, 3);
	}
	else
    if(QuadricType == 2)
	{
		gluSphere(Quadric, 0.43f, 30, 30);
	}

	for(int i=0; i<NumChildren; i++)
	{
		Children[i].UpdateAndRenderHierarchy(this);
	}

	glPopMatrix();
}

void GL_HierarchicalModel::RenderKeyPath()
{
	if(KFS.IsReady())
	{
		KFS.RenderKeyPath();
	}
}

void GL_HierarchicalModel::ResetAnimation()
{
	for(int i=0; i<NumChildren; i++)
	{
		Children[i].ResetAnimation();
	}

	if(KFS.IsReady())
	{
		KFS.Reset();
	}
}

GL_KeyFrameSystem::GL_KeyFrameSystem()
{
	NumKeyFrames = 0;
	NumHermiteCurves = 0;
	KeyFrames = NULL;
	CurrentTime = 0;
	CurrentHermiteCurve = 0;
}

GL_KeyFrameSystem::~GL_KeyFrameSystem()
{
	Destroy();
}

void GL_KeyFrameSystem::LoadKeys(char *file)
{
	VECTOR3 r, next_r;
	FILE *fp = NULL;
	int i = 0;

	Destroy();

	fp = fopen(file, "rt");
	if(fp == NULL)
	{
		fl_message("Failed to open key frame file: %s (Line: %d)", file, __LINE__);
		return;
	}

	fscanf(fp, "NumKeys: %d\n", &NumKeyFrames);

	KeyFrames = new KEY_FRAME[NumKeyFrames];
	if(KeyFrames == NULL)
	{
		fl_message("Failed to allocate key frame memory for file %s (Line: %d)", file, __LINE__);
		return;
	}

	for(i=0; i<NumKeyFrames; i++)
	{
		fscanf(fp, "Time: [%f]     Position: [%f, %f, %f]     Rotation: [%f, %f, %f]\n", 
			   &(KeyFrames[i].TimeStamp), &(KeyFrames[i].Position.x), &(KeyFrames[i].Position.y), 
			   &(KeyFrames[i].Position.z), &(KeyFrames[i].Rotation.x), &(KeyFrames[i].Rotation.y), 
			   &(KeyFrames[i].Rotation.z));
	}

	fclose(fp);


	//Generate catmul-rom curves
	i = 0;
	r = (KeyFrames[1].Position - KeyFrames[0].Position) * 0.5f;
	next_r = (KeyFrames[2].Position - KeyFrames[0].Position) * 0.5f;
	KeyFrames[i].HP.LoadHermite(KeyFrames[i].Position, KeyFrames[i + 1].Position, r, next_r);
	for(i=1; i<NumKeyFrames - 2; i++)
	{
		r = (KeyFrames[i + 1].Position - KeyFrames[i - 1].Position) * 0.5f;
		next_r = (KeyFrames[i + 2].Position - KeyFrames[i].Position) * 0.5f;
		KeyFrames[i].HP.LoadHermite(KeyFrames[i].Position, KeyFrames[i + 1].Position, r, next_r);
	}
	r = (KeyFrames[i + 1].Position - KeyFrames[i - 1].Position) * 0.5f;
	next_r = (KeyFrames[i + 1].Position - KeyFrames[i].Position) * 0.5f;
	KeyFrames[i].HP.LoadHermite(KeyFrames[i].Position, KeyFrames[i + 1].Position, r, next_r);
	

	//Generate catmul-rom curves
	i = 0;
	r = (KeyFrames[1].Rotation - KeyFrames[0].Rotation) * 0.5f;
	next_r = (KeyFrames[2].Rotation - KeyFrames[0].Rotation) * 0.5f;
	KeyFrames[i].HR.LoadHermite(KeyFrames[i].Rotation, KeyFrames[i + 1].Rotation, r, next_r);
	for(i=1; i<NumKeyFrames - 2; i++)
	{
		r = (KeyFrames[i + 1].Rotation - KeyFrames[i - 1].Rotation) * 0.5f;
		next_r = (KeyFrames[i + 2].Rotation - KeyFrames[i].Rotation) * 0.5f;
		KeyFrames[i].HR.LoadHermite(KeyFrames[i].Rotation, KeyFrames[i + 1].Rotation, r, next_r);
	}
	r = (KeyFrames[i + 1].Rotation - KeyFrames[i - 1].Rotation) * 0.5f;
	next_r = (KeyFrames[i + 1].Rotation - KeyFrames[i].Rotation) * 0.5f;
	KeyFrames[i].HR.LoadHermite(KeyFrames[i].Rotation, KeyFrames[i + 1].Rotation, r, next_r);


	for(i=0; i<NumKeyFrames-1; i++)
	{
		KeyFrames[i].du = 1.0f / (KeyFrames[i + 1].TimeStamp - KeyFrames[i].TimeStamp);
	}

	NumHermiteCurves = NumKeyFrames - 1;
}

void GL_KeyFrameSystem::Destroy(void)
{
	if(KeyFrames != NULL)
	{
		delete[] KeyFrames;
		KeyFrames = NULL;
	}

	NumKeyFrames = 0;
	NumHermiteCurves = 0;
	CurrentTime = 0;
	CurrentHermiteCurve = 0;
}

void GL_KeyFrameSystem::RenderKeyPath(void)
{

	int i = 0;
	float u = 0;
	float step = 0.01f;
	static float shimmer = 0;
	float alpha = 0;
	VECTOR3 P1, P2;

    glPushAttrib(GL_LIGHTING_BIT);
    glDisable(GL_LIGHTING);
	glDisable(GL_TEXTURE_2D);
   
	shimmer += 0.123f;
	alpha = (1.8f + (float)cos(shimmer)) / 2.4f;
	glColor4f(0.4f, 0.8f, 0.4f, alpha);
	glEnable(GL_BLEND);

	glBegin(GL_LINES); 
	for(i=0; i<NumHermiteCurves; i++)
	{
		for(u=0; u<(1.0f - step); u+=step)
		{
			P1 = KeyFrames[i].HP.ComputeCurvePoint(u);
			P2 = KeyFrames[i].HP.ComputeCurvePoint(u + step);
			glVertex3d(P1.x, P1.y, P1.z); 
			glVertex3d(P2.x, P2.y, P2.z);
		}
	}
	glEnd();

	glEnable(GL_TEXTURE_2D);
	glDisable(GL_BLEND);
	glEnable(GL_LIGHTING);
	glPopAttrib();
}

void GL_KeyFrameSystem::Reset(void)
{
	CurrentTime = 0;
	CurrentHermiteCurve = 0;
}

void GL_KeyFrameSystem::Update(float time_diff)
{
	float step = KeyFrames[CurrentHermiteCurve].du * time_diff;

	if(CurrentTime >= (1.0f - step))
	{
		CurrentHermiteCurve++;
		CurrentTime = 0;
	}
	else
	if(CurrentTime < 0.0f)
	{
		CurrentHermiteCurve--;
		CurrentTime = 1.0f;
	}

	if((CurrentHermiteCurve >= NumHermiteCurves) || (CurrentHermiteCurve < 0))
	{
		Reset();
	}

	Position = KeyFrames[CurrentHermiteCurve].HP.ComputeCurvePoint(CurrentTime);
	Rotation = KeyFrames[CurrentHermiteCurve].HR.ComputeCurvePoint(CurrentTime);

	CurrentTime += step;
}

GL_Camera::GL_Camera(GL_FrameRateTracker *timer)
{
	Timer = timer;

	Yaw = Pitch = Roll = 0;
	Position = VECTOR3(0, -8.0f, -5);
	Look_Vector = VECTOR3(0, 0, 1);
	Up_Vector = VECTOR3(0, 1, 0);
	Right_Vector = VECTOR3(1, 0, 0);
	IsAnimationPlaying = 0;
}

void GL_Camera::Move(DWORD type, float speed)
{
	if(type & CAMERA_FORWARD)
	{
		Position += Look_Vector * speed;
	}
	else
	if(type & CAMERA_RIGHT)
	{
		Position += Right_Vector * speed;
	}
	else
	if(type & CAMERA_UP)
	{
		Position += Up_Vector * speed;
	}

	if(Position.x < -9.8f) Position.x = -9.8f;
	if(Position.x >  9.8f) Position.x =  9.8f;
	if(Position.y < -9.8f) Position.y = -9.8f;
	if(Position.y >  9.8f) Position.y =  9.8f;
	if(Position.z < -9.8f) Position.z = -9.8f;
	if(Position.z >  9.8f) Position.z =  9.8f;

}

void GL_Camera::Rotate(DWORD type, float angle)
{
	if(type & CAMERA_YAW)
	{
		Yaw += angle;
	}
	else
	if(type & CAMERA_PITCH)
	{
		Pitch += angle;
	}
	else
	if(type & CAMERA_ROLL)
	{
		Roll += angle;
	}
}

void GL_Camera::SetupCamera(float width, float height)
{
    float aspect = width / height;
    glMatrixMode(GL_PROJECTION);
    gluPerspective(60, aspect, 0.1, 1000);

    glMatrixMode(GL_MODELVIEW);
}

void GL_Camera::Look(void)
{
	VECTOR3 view;
	MATRIX mat;

	if(IsAnimationPlaying)
	{
		KFS.Update(Timer->TimeDiff);
		Position = KFS.Position;
		Yaw = KFS.Rotation.x;
		Pitch = KFS.Rotation.y;
		Roll = KFS.Rotation.z;
	}
	else
	{
		if(Roll > 0)
		{
			Roll -= 0.02f;
		}

		if(Roll < 0)
		{
			Roll += 0.02f;
		}
	}

	Up_Vector = VECTOR3(0,1,0);
	Right_Vector = VECTOR3(1,0,0);
	Look_Vector = VECTOR3(0,0,1);

	mat = CreateAxisAngleRotationMat(Yaw, Up_Vector.x, Up_Vector.y, Up_Vector.z);
	Look_Vector = mat * Look_Vector;
	Right_Vector = mat * Right_Vector;	

	mat = CreateAxisAngleRotationMat(Pitch, Right_Vector.x, Right_Vector.y, Right_Vector.z);		
	Look_Vector = mat * Look_Vector;
	Up_Vector = mat * Up_Vector;

	mat = CreateAxisAngleRotationMat(Roll, Look_Vector.x, Look_Vector.y, Look_Vector.z);		
	Up_Vector = mat * Up_Vector;
	Right_Vector = mat * Right_Vector;

	view = Position + Look_Vector;

	gluLookAt(Position.x, Position.y, Position.z,	
			  view.x, view.y, view.z,	
			  Up_Vector.x, Up_Vector.y, Up_Vector.z);
}

Sphere::Sphere(const VECTOR3 &center, float radius, GL_Material *material, GL_FrameRateTracker *timer) : GL_Object(material, timer)
{
	Quadric = NULL;
	Center = center;
	Radius = radius;

	Position = VECTOR3(0, 0, 0);
}

GLUquadric *Sphere::Quadric = NULL;
void Sphere::Render()
{
    if(!Quadric) Quadric = gluNewQuadric();

	MATRIX trans_mat, scale_mat, yaw_mat, pitch_mat, roll_mat;
	MATRIX Rotation;

	yaw_mat.LoadRotationY(Yaw);
	pitch_mat.LoadRotationX(Pitch);
	roll_mat.LoadRotationY(Roll);

	Rotation = yaw_mat * pitch_mat * roll_mat;

	glPushMatrix();

	glGetFloatv(GL_MODELVIEW_MATRIX, (float *)(&Transform));
	Transform.Transpose();

	GL_Object::Render();

	trans_mat.LoadTranslation(Center.x, Center.y, Center.z);
	Transform = Transform * trans_mat;
	Transform = Transform * Rotation;
	scale_mat.LoadScaling(Scale.x, Scale.y, Scale.z);
	Transform = Transform * scale_mat;

	Transform.Transpose();
	glLoadMatrixf((float *)(&Transform));

    gluSphere(Quadric, Radius, 30, 30);
	glPopMatrix();
}

void Sphere::ComputeBoundingBox(VECTOR3 &min, VECTOR3 &max)
{
    VECTOR3 r(Radius, Radius, Radius);
    min = r * -1;
    max = r;

	glLoadMatrixf((float *)(&Transform));
}

Cylinder::Cylinder(char *file, int segments, int slices, const VECTOR3 &center, GL_Material *material, GL_FrameRateTracker *timer) : GL_Object(material, timer)
{
	char buffer[256];
	int index = 0;
	int i = 0, j = 0;
	float u = 0, v = 0, du = 0, dv = 0, radius;

	float r_gap = 0, v_gap = 0;
	int last_comp = 0, next_comp = 0;
	int num_components = 0;
	CYLINDER_COMPONENT *components = NULL;

	VECTOR3 *normals = NULL;
	VECTOR3 vSum, Vector1, Vector2;
	int num_shared = 0;

	FILE *fp = fopen(file, "rt");
	if(fp == NULL)
	{
		fl_message("Error opening file: %s", file);
		return;
	}

	while(fgets(buffer, 256, fp))
	{
		num_components++;
	}
	
	components = new CYLINDER_COMPONENT[num_components];
	if(components == NULL)
	{
		fl_message("Error allocating memory: File( %s ) Line( %d )", __FILE__, __LINE__);
		return;
	}	

	rewind(fp);
	for(i=0; i<num_components; i++)
	{
		fscanf(fp, "%f %f\n", &(components[i].v), &(components[i].radius));
		if(components[i].radius > MaxRadius)
		{
			MaxRadius = components[i].radius;
		}
	}
	
	NumVertices = segments * (slices + 1);
	NumFaces = segments * slices * 2;

	Vertices = new VERTEX[NumVertices];
	if(Vertices == NULL)
	{
		fl_message("Error allocating memory: File( %s ) Line( %d )", __FILE__, __LINE__);
		return;
	}

	Faces = new FACE[NumFaces];
	if(Faces == NULL)
	{
		fl_message("Error allocating memory: File( %s ) Line( %d )", __FILE__, __LINE__);
		return;
	}

	Center = center;
	Height = 1;
	MaxRadius = 1;
	Position = VECTOR3(0, 0, 0);
	du = 360.0f / (float)segments;
	dv = Height / (float)slices;

	index = 0;
	for(i=0; i<(slices + 1); i++)
	{
		last_comp = next_comp = -1;
		for(j=0; j<num_components; j++)
		{
			if(v >= components[j].v) last_comp = j;
		}
		next_comp = last_comp + 1;

		if(last_comp == -1)
		{
			r_gap = components[next_comp].radius - 0;
			v_gap = components[next_comp].v - 0;
	
			radius = (v - 0) * (r_gap / v_gap) + 0;
		}
		else
		{
			if(next_comp > (num_components - 1))
			{
				r_gap = 0 - components[last_comp].radius;
				v_gap = Height - components[last_comp].v;

				radius = (v - components[last_comp].v) * (r_gap / v_gap) + components[last_comp].radius;
			}
			else
			{
				r_gap = components[next_comp].radius - components[last_comp].radius;
				v_gap = components[next_comp].v - components[last_comp].v;

				radius = (v - components[last_comp].v) * (r_gap / v_gap) + components[last_comp].radius;
			}
		}

		for(u = 0; u < 360.0f; u += du)
		{
			Vertices[index].Position.x = radius * (float)cos(u * (PI / 180.0f));
			Vertices[index].Position.y = radius * (float)sin(u * (PI / 180.0f));
			Vertices[index].Position.z = v;
			index++;
		}
		v += dv;
	}

	for(i=0; i<(NumFaces / 2); i++)
	{
		Faces[2*i].a = i;

		if(((i + 1) % segments) == 0)
		{
			Faces[2*i].b = (i + 1) - segments;
		}
		else
		{
			Faces[2*i].b = (i + 1);
		}

		Faces[2*i].c = i + segments;


		Faces[2*i + 1].a = Faces[2*i].b;
		Faces[2*i + 1].b = Faces[2*i].c;
		
		
		if(((Faces[2*i].c + 1) % segments) == 0)
		{
			Faces[2*i + 1].c = (Faces[2*i].c + 1) - segments;
		}
		else
		{
			Faces[2*i + 1].c = Faces[2*i].c + 1;
		}
	}


	num_shared = 0;
	vSum = VECTOR3(0, 0, 0);
	normals = new VECTOR3[NumFaces];
	if(normals == NULL)
	{
		fl_message("Error allocating memory: File( %s ) Line( %d )", __FILE__, __LINE__);
		return;
	}

	for(i=0; i<NumFaces; i++)
	{
		Vector1 = Vertices[Faces[i].a].Position - Vertices[Faces[i].c].Position;
		Vector2 = Vertices[Faces[i].c].Position - Vertices[Faces[i].b].Position;

		normals[i] = Vec3CrossProduct(Vector1, Vector2);		
	}

	for(i=0; i<NumVertices; i++)
	{
		for(j=0; j<NumFaces; j++)
		{
			if((Faces[j].a == i) || (Faces[j].b == i) || (Faces[j].c == i))
			{
				vSum += normals[j];
				num_shared++;
			}
		}

		Vertices[i].Normal.x = (vSum.x / (float)(-num_shared));
		Vertices[i].Normal.y = (vSum.y / (float)(-num_shared));
		Vertices[i].Normal.z = (vSum.z / (float)(-num_shared));
		Vertices[i].Normal.Normalize();

		vSum = VECTOR3(0, 0, 0);
		num_shared = 0;
	}

	if(components != NULL) delete[] components;
	if(normals != NULL) delete[] normals;

	fclose(fp);

}

Cylinder::~Cylinder()
{
	if(Faces != NULL)
	{
		delete[] Faces;
		Faces = NULL;
	}

	if(Vertices != NULL)
	{
		delete[] Vertices;
		Vertices = NULL;
	}
}

void Cylinder::Render()
{
    if((Vertices == NULL) || (Faces == NULL)) return;

	MATRIX trans_mat, scale_mat, yaw_mat, pitch_mat, roll_mat;
	MATRIX Rotation;

	glPushMatrix();

	glEnable(GL_LIGHTING);
	glDisable(GL_TEXTURE_2D);

	glGetFloatv(GL_MODELVIEW_MATRIX, (float *)(&Transform));
	Transform.Transpose();

	GL_Object::Render();

	Up_Vector = VECTOR3(0,1,0);
	Right_Vector = VECTOR3(1,0,0);
	Look_Vector = VECTOR3(0,0,1);

	yaw_mat = CreateAxisAngleRotationMat(Yaw, Up_Vector.x, Up_Vector.y, Up_Vector.z);
	Look_Vector = yaw_mat * Look_Vector;
	Right_Vector = yaw_mat * Right_Vector;	

	pitch_mat = CreateAxisAngleRotationMat(Pitch, Right_Vector.x, Right_Vector.y, Right_Vector.z);		
	Look_Vector = pitch_mat * Look_Vector;
	Up_Vector = pitch_mat * Up_Vector;

	roll_mat = CreateAxisAngleRotationMat(Roll, Look_Vector.x, Look_Vector.y, Look_Vector.z);		
	Up_Vector = roll_mat * Up_Vector;
	Right_Vector = roll_mat * Right_Vector;

	Rotation = yaw_mat * pitch_mat * roll_mat;

//	trans_mat.LoadTranslation(Center.x, Center.y, Center.z);
	Transform = Transform * trans_mat;
	Transform = Transform * Rotation;
	scale_mat.LoadScaling(Scale.x, Scale.y, Scale.z);
	Transform = Transform * scale_mat;

	Transform.Transpose();
	glLoadMatrixf((float *)(&Transform));

    glBegin(GL_TRIANGLES);
	for(int i=0; i<NumFaces; i++)
	{
		glNormal3f(Vertices[Faces[i].a].Normal.x, Vertices[Faces[i].a].Normal.y, Vertices[Faces[i].a].Normal.z);		
		glVertex3f(Vertices[Faces[i].a].Position.x, Vertices[Faces[i].a].Position.y, Vertices[Faces[i].a].Position.z);
		
		glNormal3f(Vertices[Faces[i].b].Normal.x, Vertices[Faces[i].b].Normal.y, Vertices[Faces[i].b].Normal.z);		
		glVertex3f(Vertices[Faces[i].b].Position.x, Vertices[Faces[i].b].Position.y, Vertices[Faces[i].b].Position.z);
		
		glNormal3f(Vertices[Faces[i].c].Normal.x, Vertices[Faces[i].c].Normal.y, Vertices[Faces[i].c].Normal.z);		
		glVertex3f(Vertices[Faces[i].c].Position.x, Vertices[Faces[i].c].Position.y, Vertices[Faces[i].c].Position.z);				
	}
    glEnd();	

	glEnable(GL_TEXTURE_2D);

	glPopMatrix();
}

void Cylinder::ComputeBoundingBox(VECTOR3 &min, VECTOR3 &max)
{
	MATRIX transform, trans_mat, scale_mat;
    VECTOR3 r(MaxRadius, MaxRadius, (Height / 2.0f));
    min = r * -1;
	min.z += (Height / 2.0f);
    max = r;
	max.z += (Height / 2.0f);

	glLoadMatrixf((float *)(&Transform));
}

GL_Level::GL_Level()
{
	Material.Diffuse = COLOR(110, 120, 160, 255);
	Material.Specular = Material.Diffuse;
	Material.Ambient = Material.Diffuse;
	Material.Power = 14.12f;	
}

void GL_Level::Render()
{	
	glPushMatrix();

	glEnable(GL_LIGHTING);
	glEnable(GL_TEXTURE_2D);
	glDisable(GL_BLEND);

	Material.SetMaterial();

	glBindTexture(GL_TEXTURE_2D, 3);

	glScalef(10.0f, 10.0f, 10.0f);

	glBegin(GL_QUADS);
		//Render Front Face
		glNormal3f( 0.0f, 0.0f, -1.0f);
		glTexCoord2f(0.0f, 0.0f); glVertex3f(-1.0f, -1.0f,  1.0f);
		glTexCoord2f(1.0f, 0.0f); glVertex3f( 1.0f, -1.0f,  1.0f);
		glTexCoord2f(1.0f, 1.0f); glVertex3f( 1.0f,  1.0f,  1.0f);
		glTexCoord2f(0.0f, 1.0f); glVertex3f(-1.0f,  1.0f,  1.0f);
		
		//Render Back Face
		glNormal3f( 0.0f, 0.0f,1.0f);
		glTexCoord2f(1.0f, 0.0f); glVertex3f(-1.0f, -1.0f, -1.0f);
		glTexCoord2f(1.0f, 1.0f); glVertex3f(-1.0f,  1.0f, -1.0f);
		glTexCoord2f(0.0f, 1.0f); glVertex3f( 1.0f,  1.0f, -1.0f);
		glTexCoord2f(0.0f, 0.0f); glVertex3f( 1.0f, -1.0f, -1.0f);
		
		//Render Top Face
		glNormal3f( 0.0f, -1.0f, 0.0f);
		glTexCoord2f(0.0f, 1.0f); glVertex3f(-1.0f,  1.0f, -1.0f);
		glTexCoord2f(0.0f, 0.0f); glVertex3f(-1.0f,  1.0f,  1.0f);
		glTexCoord2f(1.0f, 0.0f); glVertex3f( 1.0f,  1.0f,  1.0f);
		glTexCoord2f(1.0f, 1.0f); glVertex3f( 1.0f,  1.0f, -1.0f);
		
		//Render Bottom Face
		glNormal3f( 0.0f,1.0f, 0.0f);
		glTexCoord2f(1.0f, 1.0f); glVertex3f(-1.0f, -1.0f, -1.0f);
		glTexCoord2f(0.0f, 1.0f); glVertex3f( 1.0f, -1.0f, -1.0f);
		glTexCoord2f(0.0f, 0.0f); glVertex3f( 1.0f, -1.0f,  1.0f);
		glTexCoord2f(1.0f, 0.0f); glVertex3f(-1.0f, -1.0f,  1.0f);
		
		//Render Right face
		glNormal3f(-1.0f, 0.0f, 0.0f);
		glTexCoord2f(1.0f, 0.0f); glVertex3f( 1.0f, -1.0f, -1.0f);
		glTexCoord2f(1.0f, 1.0f); glVertex3f( 1.0f,  1.0f, -1.0f);
		glTexCoord2f(0.0f, 1.0f); glVertex3f( 1.0f,  1.0f,  1.0f);
		glTexCoord2f(0.0f, 0.0f); glVertex3f( 1.0f, -1.0f,  1.0f);
		
		//Render Left Face
		glNormal3f(1.0f, 0.0f, 0.0f);
		glTexCoord2f(0.0f, 0.0f); glVertex3f(-1.0f, -1.0f, -1.0f);
		glTexCoord2f(1.0f, 0.0f); glVertex3f(-1.0f, -1.0f,  1.0f);
		glTexCoord2f(1.0f, 1.0f); glVertex3f(-1.0f,  1.0f,  1.0f);
		glTexCoord2f(0.0f, 1.0f); glVertex3f(-1.0f,  1.0f, -1.0f);
	glEnd();

	glBindTexture(GL_TEXTURE_2D, 0);

	glDisable(GL_TEXTURE_2D);

	glPopMatrix();
}

GL_ParticleSystem::GL_ParticleSystem()
{
	Camera = NULL;
	NumParticles = 0;
	NumParticlesActive = 0;
	Particles = NULL;
	IsParticleEngineActive = 1;
}

GL_ParticleSystem::~GL_ParticleSystem()
{
	Destroy();
}

int GL_ParticleSystem::Initialize(int num_particles)
{
	int i = 0;

	NumParticles = num_particles;
	NumParticlesActive = 0;
	Particles = new PARTICLE[NumParticles];
	if(Particles == NULL) return 0;

	for(i=0; i<NumParticles; i++)
	{
		memset(&(Particles[i]), 0, sizeof(PARTICLE));
	}

	NumSpringPairs = num_particles;
	NumSpringPairsActive = 0;
	SpringPairs = new SPRING_PAIR[NumSpringPairs];
	if(SpringPairs == NULL) return 0;

	for(i=0; i<NumSpringPairs; i++)
	{
		memset(&(SpringPairs[i]), 0, sizeof(SPRING_PAIR));
	}

	return 1;
}

void GL_ParticleSystem::Destroy(void)
{
	Camera = NULL;

	NumParticles = 0;
	NumParticlesActive = 0;

	if(Particles != NULL)
	{
		delete[] Particles;
		Particles = NULL;
	}
}

void GL_ParticleSystem::ConnectPair(int index1, int index2, float rest_length, float k)
{
	if(!IsParticleEngineActive) return;

	if((Particles == NULL) || (SpringPairs == NULL)) return;

	int index = 0;
	static int last_index = -1;

	for(index=(last_index+1); index<NumSpringPairs; index++)
	{
		if(SpringPairs[index].IsActive == 0)
		{
			SpringPairs[index].IsActive = 1;
			SpringPairs[index].Index1 = index1;
			SpringPairs[index].Index2 = index2;
			SpringPairs[index].RestLength = rest_length;
			SpringPairs[index].K = k;

			last_index = index;
			NumSpringPairsActive++;
			return;
		}
	}

	for(index=0; index<(last_index+1); index++)
	{
		if(SpringPairs[index].IsActive == 0)
		{
			SpringPairs[index].IsActive = 1;
			SpringPairs[index].Index1 = index1;
			SpringPairs[index].Index2 = index2;
			SpringPairs[index].RestLength = rest_length;
			SpringPairs[index].K = k;

			last_index = index;
			NumSpringPairsActive++;
			return;
		}
	}

	SPRING_PAIR *tmp_list = NULL;
	tmp_list = new SPRING_PAIR[NumSpringPairs * 2];
	if(tmp_list == NULL) 
	{
		fl_message("WARNING: Unable to allocate particle engine spring pair memory (Line: %d)", __LINE__);
		return;
	}
	
	memcpy(tmp_list, SpringPairs, sizeof(SPRING_PAIR) * NumSpringPairs);
	delete[] SpringPairs;
	SpringPairs = tmp_list;
	tmp_list = NULL;
	
	for(int i=NumSpringPairs; i<(NumSpringPairs * 2); i++)
	{
		memset(&(SpringPairs[i]), 0, sizeof(SPRING_PAIR));
	}

	SpringPairs[NumSpringPairs].IsActive = 1;
	SpringPairs[NumSpringPairs].Index1 = index1;
	SpringPairs[NumSpringPairs].Index2 = index2;
	SpringPairs[NumSpringPairs].RestLength = rest_length;
	SpringPairs[NumSpringPairs].K = k;
	
	last_index = NumSpringPairs;
	NumSpringPairsActive++;
	NumSpringPairs *= 2;	
}

int	GL_ParticleSystem::AddParticle(int tex_index, float x, float y, float z, 
								   float xv, float yv, float zv, float scale, 
								   COLOR color, int max_life, float mass)
{
	if(!IsParticleEngineActive) return 0;

	if(Particles == NULL) return 0;

	int index = 0, ret_index = 0;
	static int last_index = -1;

	for(index=(last_index+1); index<NumParticles; index++)
	{
		if(Particles[index].IsActive == 0)
		{
			Particles[index].IsActive = 1;
			Particles[index].TextureIndex = tex_index;
			Particles[index].Pos = VECTOR3(x, y, z);
			Particles[index].LastPos = VECTOR3(x, y, z);
			Particles[index].Velocity = VECTOR3(xv, yv, zv);
			Particles[index].Scale = scale;
			Particles[index].Color = color;
			Particles[index].MaxLife = max_life;
			Particles[index].Age = 0;
			Particles[index].Mass = mass;

			ret_index = index;
			last_index = index;
			NumParticlesActive++;
			return ret_index;
		}
	}

	for(index=0; index<(last_index+1); index++)
	{
		if(Particles[index].IsActive == 0)
		{
			Particles[index].IsActive = 1;
			Particles[index].TextureIndex = tex_index;
			Particles[index].Pos = VECTOR3(x, y, z);
			Particles[index].LastPos = VECTOR3(x, y, z);
			Particles[index].Velocity = VECTOR3(xv, yv, zv);
			Particles[index].Scale = scale;
			Particles[index].Color = color;
			Particles[index].MaxLife = max_life;
			Particles[index].Age = 0;
			Particles[index].Mass = mass;

			ret_index = index;
			last_index = index;
			NumParticlesActive++;
			return ret_index;
		}
	}

	PARTICLE *tmp_list = NULL;
	tmp_list = new PARTICLE[NumParticles * 2];
	if(tmp_list == NULL) 
	{
		fl_message("WARNING: Unable to allocate particle engine memory (Line: %d)", __LINE__);
		return 0;
	}
	
	memcpy(tmp_list, Particles, sizeof(PARTICLE) * NumParticles);
	delete[] Particles;
	Particles = tmp_list;
	tmp_list = NULL;
	
	for(int i=NumParticles; i<(NumParticles * 2); i++)
	{
		memset(&(Particles[i]), 0, sizeof(PARTICLE));
	}

	Particles[NumParticles].IsActive = 1;
	Particles[NumParticles].TextureIndex = tex_index;
	Particles[NumParticles].Pos = VECTOR3(x, y, z);
	Particles[NumParticles].LastPos = VECTOR3(x, y, z);
	Particles[NumParticles].Velocity = VECTOR3(xv, yv, zv);
	Particles[NumParticles].Scale = scale;
	Particles[NumParticles].Color = color;
	Particles[NumParticles].MaxLife = max_life;
	Particles[NumParticles].Age = 0;
	Particles[NumParticles].Mass = mass;	

	ret_index = NumParticles;

	last_index = NumParticles;
	NumParticlesActive++;
	NumParticles *= 2;

	return ret_index;
}

void GL_ParticleSystem::KillParticles(void)
{
	memset(Particles, 0, sizeof(PARTICLE) * NumParticles);
	NumParticlesActive = 0;
}

void GL_ParticleSystem::Update(void)
{
	if(!IsParticleEngineActive) return;

	int i = 0;
	float distance = 0;
	VECTOR3 d, force, accel;

	for(i=0; i<NumParticles; i++)
	{
		if(Particles[i].IsActive)
		{
			Particles[i].LastPos = Particles[i].Pos;
			Particles[i].Pos += Particles[i].Velocity;
			
			//I know it's not realistic, but I decided to use the 
			//Particle mass as an adjustment factor in the gravitational
			//acceleration calculation. This way weapon particles can 
			//fall at a different rate than fountain particles. The only
			//alternative was to increase weapon particle velocity, which
			//didn't look nearly as good.
			Particles[i].Velocity.y -= (Particles[i].Mass / 10000.0f);

			if(Particles[i].Pos.x < -9.99f) Particles[i].Velocity.x =  ABS(Particles[i].Velocity.x * 0.7f);
			if(Particles[i].Pos.x >  9.99f) Particles[i].Velocity.x = -ABS(Particles[i].Velocity.x * 0.7f);
			if(Particles[i].Pos.y < -9.99f) Particles[i].Velocity.y =  ABS(Particles[i].Velocity.y * 0.7f);
			if(Particles[i].Pos.y >  9.99f) Particles[i].Velocity.y = -ABS(Particles[i].Velocity.y * 0.7f);
			if(Particles[i].Pos.z < -9.99f) Particles[i].Velocity.z =  ABS(Particles[i].Velocity.z * 0.7f);
			if(Particles[i].Pos.z >  9.99f) Particles[i].Velocity.z = -ABS(Particles[i].Velocity.z * 0.7f);

			if(Particles[i].MaxLife != -1)
			{
				Particles[i].Age++;
				if(Particles[i].Age >= Particles[i].MaxLife)
				{
					Particles[i].IsActive = 0;
					NumParticlesActive--;
				}
			}
		}
	}

	for(i=0; i<NumSpringPairs; i++)
	{
		if(SpringPairs[i].IsActive)
		{
			if(Particles[SpringPairs[i].Index1].IsActive && 
			   Particles[SpringPairs[i].Index2].IsActive)
			{
				d = Particles[SpringPairs[i].Index1].Pos - Particles[SpringPairs[i].Index2].Pos;
				distance = d.Length();
				d.Normalize();

				force = d * (-SpringPairs[i].K * (distance - SpringPairs[i].RestLength));
				
				accel = force / Particles[SpringPairs[i].Index1].Mass;
				Particles[SpringPairs[i].Index1].Velocity += accel;

				accel = force / -Particles[SpringPairs[i].Index2].Mass;
				Particles[SpringPairs[i].Index2].Velocity += accel;
			}
		}
	}
}

void GL_ParticleSystem::Render(void)
{
	if(!IsParticleEngineActive) return;

	if((Camera == NULL) || (Particles == NULL)) return;

	int i = 0;
	VECTOR3 p;
	MATRIX trans_mat, scale_mat;
	MATRIX original, world_mat;

	glPushAttrib(GL_LIGHTING_BIT);
	glPushMatrix();

	glGetFloatv(GL_MODELVIEW_MATRIX, (float *)(&original));
	original.Transpose();

	glBlendFunc(GL_SRC_ALPHA, GL_ONE);

	glDisable (GL_LIGHTING);
    glDisable(GL_DEPTH_TEST);
	glEnable(GL_BLEND);
	glEnable(GL_TEXTURE_2D);

	for(i=0; i<NumParticles; i++)
	{
		if(Particles[i].IsActive)
		{
			trans_mat.LoadTranslation(Particles[i].Pos.x, Particles[i].Pos.y, Particles[i].Pos.z);
			scale_mat.LoadScaling(Particles[i].Scale, Particles[i].Scale, Particles[i].Scale);
			
			world_mat = original * trans_mat * scale_mat;
			world_mat.Transpose();		
			glLoadMatrixf((float *)(&world_mat));

			glColor4f(Particles[i].Color.r, Particles[i].Color.g, Particles[i].Color.b, Particles[i].Color.a);
			glBindTexture(GL_TEXTURE_2D, Particles[i].TextureIndex);

			glBegin(GL_TRIANGLE_STRIP);

			glTexCoord2f(0.0f, 1.0f);
			p = (Camera->Up_Vector * 0.5f) - (Camera->Right_Vector * 0.5f);
			glVertex3f(p.x, p.y, p.z);

			glTexCoord2f(1.0f, 1.0f);
			p = (Camera->Up_Vector * 0.5f) + (Camera->Right_Vector * 0.5f);
			glVertex3f(p.x, p.y, p.z);

			glTexCoord2f(0.0f, 0.0f);
			p = ((Camera->Up_Vector * 0.5f) + (Camera->Right_Vector * 0.5f)) * -1.0f;
			glVertex3f(p.x, p.y, p.z);

			glTexCoord2f(1.0f, 0.0f);
			p = (Camera->Up_Vector * 0.5f) * -1.0f + (Camera->Right_Vector * 0.5f);
			glVertex3f(p.x, p.y, p.z);

			glEnd();				
		}
	}                        

	glBindTexture(GL_TEXTURE_2D, 0);
	glDisable(GL_TEXTURE_2D);

	glEnable(GL_DEPTH_TEST);	

	glPopMatrix();

    glColor4f(0.2f, 0.2f, 1.0f, 0.5f);
	for(i=0; i<NumSpringPairs; i++)
	{
		if(SpringPairs[i].IsActive)
		{
			if((!Particles[SpringPairs[i].Index1].IsActive) || 
			   (!Particles[SpringPairs[i].Index2].IsActive))
			{
				SpringPairs[i].IsActive = 0;
				continue;
			}

			glBegin(GL_LINES);
			glVertex3d(Particles[SpringPairs[i].Index1].Pos.x, 
					   Particles[SpringPairs[i].Index1].Pos.y,
					   Particles[SpringPairs[i].Index1].Pos.z);
			glVertex3d(Particles[SpringPairs[i].Index2].Pos.x, 
					   Particles[SpringPairs[i].Index2].Pos.y,
					   Particles[SpringPairs[i].Index2].Pos.z);				
			glEnd();
		}
	}	


	glDisable(GL_BLEND);
	glEnable(GL_LIGHTING);
	glPopAttrib();
}


GL_FrameRateTracker::GL_FrameRateTracker()
{
	ProgramStartTime = 0;
	StartTime = 0;
	FrameCount = 0;
	LastTime = 0;
	CounterFrequency = 0;
	TimeDiff = 0;
	FPS = 0;	
}

void GL_FrameRateTracker::Initialize(void)
{
	QueryPerformanceFrequency((LARGE_INTEGER*)&CounterFrequency);
	QueryPerformanceCounter((LARGE_INTEGER*)&ProgramStartTime);
}

void GL_FrameRateTracker::Update(void)
{
	__int64 time = 0;
	double fps_time_diff = 0;
	if(LastTime == 0)
	{
		QueryPerformanceCounter((LARGE_INTEGER*)&LastTime);
	}

	QueryPerformanceCounter((LARGE_INTEGER*)&time);
	TimeDiff = (float)((double)(time - LastTime) / (double)CounterFrequency);
	LastTime = time;
	
	fps_time_diff = (double)(time - StartTime) / (double)CounterFrequency;
	if((StartTime == 0) || (fps_time_diff > 0.250)) 
	{
		FPS = (float)((double)FrameCount / fps_time_diff);
		FrameCount = 0;
		StartTime = time;
	}
	FrameCount++;
}


