/*
 * AI system - enemy movement AI and special effects
 * Converted from x86 assembly
 */

#include "ai.h"
#include "enemy.h"
#include "player.h"
#include "ppe.h"
#include "rand.h"
#include "defs.h"
#include <math.h>

/* AI constants */
#define G 450.0f
#define INNER_RADIUS_CUTOFF_SQ (INNER_RADIUS_CUTOFF * INNER_RADIUS_CUTOFF)
#define OUTER_RADIUS_CUTOFF_SQ (OUTER_RADIUS_CUTOFF * OUTER_RADIUS_CUTOFF)

extern float SIN_LOOK[256];
extern float COS_LOOK[256];

/* Enemy firing rate constants - mirrors assembly ai.asm:97-101 */
float fltEnemyFiringRate[5] = {12.6f, 14.5331f, 16.31f, 18.2f, 7.3112f};

/* Particle speed constants - mirrors assembly ai.asm:103-107 */
float fltPSpeed1 = 4.2f;
float fltPSpeed2 = 5.33f;
float fltPSpeed3 = 6.112f;
float fltPSpeed4 = 7.5f;
float fltPSpeed5 = 8.82f;

/*
 * AI movement for enemies - gravitational attraction to player
 */
void EnemyMove(void) {
    extern float fltPlayerMass;
    float g_mass = fltPlayerMass * G;

    for (int i = 0; i < 100; i++) {
        if (Enemies[i].enemy_type == (uint32_t)-1) break;
        if (Enemies[i].enemy_active == -1) continue;

        /* Calculate delta X considering map wrap - mirrors assembly lines 153-175 */
        float delta_x1_flt = fltPlayerX - Enemies[i].enemy_x_float;
        int delta_x1_int = lrintf(delta_x1_flt);  /* fistp on line 161 */

        float delta_x2_flt;
        int delta_x2_int;

        /* Assembly checks the INT version (line 164-166) */
        if (delta_x1_int >= 0) {
            delta_x2_flt = delta_x1_flt - MAP_WIDTH;  /* Line 171 */
        } else {
            delta_x2_flt = delta_x1_flt + MAP_WIDTH;  /* Line 168 */
        }
        delta_x2_int = (int)lrintf(fabsf(delta_x2_flt));  /* fabs + fistp on lines 174-175 */

        /* Choose shortest distance - mirrors assembly lines 279-288 */
        /* NOTE: Must compare absolute values, not signed vs unsigned! */
        int delta_x1_abs = (delta_x1_int < 0) ? -delta_x1_int : delta_x1_int;
        float delta_x = (delta_x1_abs < delta_x2_int) ? delta_x1_flt : delta_x2_flt;

        /* Calculate delta Y considering map wrap - mirrors assembly lines 179-201 */
        float delta_y1_flt = fltPlayerY - Enemies[i].enemy_y_float;
        int delta_y1_int = lrintf(delta_y1_flt);  /* fistp on line 187 */

        float delta_y2_flt;
        int delta_y2_int;

        /* Assembly checks the INT version (line 190-192) */
        if (delta_y1_int >= 0) {
            delta_y2_flt = delta_y1_flt - MAP_HEIGHT;  /* Line 197 */
        } else {
            delta_y2_flt = delta_y1_flt + MAP_HEIGHT;  /* Line 194 */
        }
        delta_y2_int = (int)lrintf(fabsf(delta_y2_flt));  /* fabs + fistp on lines 200-201 */

        /* Choose shortest distance - mirrors assembly lines 290-299 */
        /* NOTE: Must compare absolute values, not signed vs unsigned! */
        int delta_y1_abs = (delta_y1_int < 0) ? -delta_y1_int : delta_y1_int;
        float delta_y = (delta_y1_abs < delta_y2_int) ? delta_y1_flt : delta_y2_flt;

        /* Calculate distance squared - mirrors assembly lines 301-312 */
        float radius_sq = delta_x * delta_x + delta_y * delta_y;
        int radius_int = lrintf(sqrtf(radius_sq));  /* Matches fistp at line 312 */

        /* Skip if too close - mirrors assembly lines 429-431 */
        /* OUTER_RADIUS_CUTOFF check is commented out in assembly (lines 450-451)! */
        if (radius_int < INNER_RADIUS_CUTOFF) {
            continue;
        }

        /* Calculate gravitational force - mirrors assembly lines 399-404 */
        /* Assembly: fld [_RadiusFlt] loads R^2, then divides G_Mass by R^2 */
        float radius = sqrtf(radius_sq);
        float force = g_mass / radius_sq;  /* Force = G*M / R^2 */

        /* Calculate acceleration components - mirrors assembly lines 412-427 */
        /* Assembly normalizes delta by dividing by R, then multiplies by force */
        float norm_x = delta_x / radius;  /* Normalized X direction */
        float norm_y = delta_y / radius;  /* Normalized Y direction */
        float acc_x = norm_x * force;
        float acc_y = norm_y * force;

        /* Apply acceleration to velocity - mirrors assembly lines 433-441 */
        float new_vel_x = Enemies[i].enemy_x_vel_float + acc_x;
        float new_vel_y = Enemies[i].enemy_y_vel_float + acc_y;

        /* Debug gravity and velocity limiting every 120 frames for first enemy */
        static int gravity_debug = 0;
        if (i == 0 && gravity_debug++ % 120 == 0) {
            printf("Enemy %d: pos=(%.1f,%.1f) player=(%.1f,%.1f) delta=(%.1f,%.1f)\n",
                   i, Enemies[i].enemy_x_float, Enemies[i].enemy_y_float,
                   fltPlayerX, fltPlayerY, delta_x, delta_y);
            printf("  radius=%.1f, force=%.3f, acc=(%.3f,%.3f)\n",
                   radius, force, acc_x, acc_y);
            printf("  old_vel=(%.3f,%.3f) new_vel=(%.3f,%.3f)\n",
                   Enemies[i].enemy_x_vel_float, Enemies[i].enemy_y_vel_float,
                   new_vel_x, new_vel_y);
        }

        /* Limit velocity COMPONENT-WISE (not magnitude!) */
        /* NOTE: Assembly has bug - only limits positive velocities (uses jg without abs)
         * This causes negative velocities to grow unbounded. Fixed here to use abs() */
        int vel_x_int = lrintf(fabsf(new_vel_x));  /* Use absolute value */
        if (vel_x_int <= MAX_ENEMY_SPEED) {
            Enemies[i].enemy_x_vel_float = new_vel_x;
        }

        int vel_y_int = lrintf(fabsf(new_vel_y));  /* Use absolute value */
        if (vel_y_int <= MAX_ENEMY_SPEED) {
            Enemies[i].enemy_y_vel_float = new_vel_y;
        }

        /* Calculate angle to player - mirrors assembly lines 340-346 */
        /* Assembly uses fpatan then multiplies by fltRadToDeg256 and uses fistp for rounding */
        float angle_rad = atan2f(delta_y, delta_x);
        extern float fltRadToDeg256;
        uint8_t target_angle = (uint8_t)lrintf(angle_rad * fltRadToDeg256);

        /* Enemy turning rate limiter - mirrors assembly lines 357-393 */
        uint8_t current_angle = Enemies[i].enemy_angle;
        int angle_diff = (int)current_angle - (int)target_angle;

        if (angle_diff > MAX_ENEMY_TURNING_SPEED) {
            /* Turn clockwise toward target */
            Enemies[i].enemy_angle = current_angle - MAX_ENEMY_TURNING_SPEED;
        } else if (angle_diff < -MAX_ENEMY_TURNING_SPEED) {
            /* Turn counter-clockwise toward target */
            Enemies[i].enemy_angle = current_angle + MAX_ENEMY_TURNING_SPEED;
        } else {
            /* Close enough, snap to target angle */
            Enemies[i].enemy_angle = target_angle;
        }

        /* Enemy firing logic - mirrors assembly lines 478-516 */
        if ((int)radius < 400) {  /* Boss firing radius: 400 (not 400*400!) - mirrors line 478 */
            if (Enemies[i].enemy_size == 1) {
                /* Boss weapons - mirrors assembly lines 536-668 */
                static int fire_debug_count = 0;
                if (fire_debug_count++ % 60 == 0) {  /* Debug every 60 frames */
                    printf("Boss %d in range! type=%d, radius=%d\n", i, Enemies[i].enemy_type, (int)radius);
                }
                if (Enemies[i].enemy_type != 3) {
                    /* Regular bosses (types 0, 1, 2) - mirrors assembly lines 540-604 */
                    uint32_t fire_check = Rand() % 100;
                    /* Assembly uses < 60, NOT >= 60! (line 544-545: cmp edx,60 / jae near .NoFire) */
                    if (fire_check < 60) {
                        uint8_t angle = Enemies[i].enemy_angle;

                        /* All boss types fire main weapon - mirrors lines 548-551 */
                        AddParticle(3, LARGE_PARTICLE, 5, Enemies[i].enemy_x_float, Enemies[i].enemy_y_float,
                                  angle, fltEnemyFiringRate[1], 30, 4);

                        /* Boss type 1+ fires 3 additional weapons - mirrors lines 554-576 */
                        if (Enemies[i].enemy_type >= 1) {
                            AddParticle(3, LARGE_PARTICLE, 7, Enemies[i].enemy_x_float, Enemies[i].enemy_y_float,
                                      angle + 10, fltEnemyFiringRate[1], 30, 6);
                            AddParticle(3, LARGE_PARTICLE, 7, Enemies[i].enemy_x_float, Enemies[i].enemy_y_float,
                                      angle - 10, fltEnemyFiringRate[1], 30, 6);
                            AddParticle(3, SMALL_PARTICLE, 8, Enemies[i].enemy_x_float, Enemies[i].enemy_y_float,
                                      angle, fltEnemyFiringRate[3], 30, 10);
                        }

                        /* Boss type 2+ has 4% chance for shockwave - mirrors lines 578-603 */
                        if (Enemies[i].enemy_type >= 2) {
                            uint32_t shock_check = Rand() % 1000;
                            if (shock_check < 40) {  /* 4% chance */
                                /* Fire 256 particles in a circle - mirrors lines 592-600 */
                                for (int shock_angle = 0; shock_angle < 256; shock_angle++) {
                                    AddParticle(3, SMALL_PARTICLE, 8, Enemies[i].enemy_x_float, Enemies[i].enemy_y_float,
                                              shock_angle, fltEnemyFiringRate[4], 100, 1);
                                }
                            }
                        }
                    }
                } else {
                    /* SHIMDOG special boss (type 3) - mirrors assembly lines 607-668 */
                    uint32_t fire_check = Rand() % 100;
                    /* Assembly uses < 20, NOT >= 60! (line 614-615: cmp edx,20 / jae near .NoFire) */
                    if (fire_check < 20) {
                        uint8_t angle = Enemies[i].enemy_angle;
                        /* Fire 4 projectiles at cardinal directions - mirrors lines 618-644 */
                        AddParticle(3, LARGE_PARTICLE, 9, Enemies[i].enemy_x_float, Enemies[i].enemy_y_float,
                                  angle, fltEnemyFiringRate[0], 30, 10);
                        AddParticle(3, LARGE_PARTICLE, 9, Enemies[i].enemy_x_float, Enemies[i].enemy_y_float,
                                  angle + 64, fltEnemyFiringRate[0], 30, 10);
                        AddParticle(3, LARGE_PARTICLE, 9, Enemies[i].enemy_x_float, Enemies[i].enemy_y_float,
                                  angle + 128, fltEnemyFiringRate[0], 30, 10);
                        AddParticle(3, LARGE_PARTICLE, 9, Enemies[i].enemy_x_float, Enemies[i].enemy_y_float,
                                  angle - 64, fltEnemyFiringRate[0], 30, 10);

                        /* SHIMDOG has 6% chance for shockwave - mirrors lines 647-665 */
                        uint32_t shock_check = Rand() % 1000;
                        if (shock_check < 60) {  /* 6% chance */
                            for (int shock_angle = 0; shock_angle < 256; shock_angle++) {
                                AddParticle(3, LARGE_PARTICLE, 9, Enemies[i].enemy_x_float, Enemies[i].enemy_y_float,
                                          shock_angle, fltEnemyFiringRate[4], 100, 1);
                            }
                        }
                    }
                }
            } else {
                /* Regular enemy weapons - fire about 50% of the time - mirrors assembly lines 485-513 */
                uint32_t fire_check = Rand() & 0xFF;
                if (fire_check >= 0x7F) {
                    /* Calculate firing velocity: enemy velocity + firing direction */
                    uint8_t angle_idx = Enemies[i].enemy_angle;
                    float fire_x_vel = Enemies[i].enemy_x_vel_float +
                                     (COS_LOOK[angle_idx] * fltEnemyFiringRate[0]);
                    float fire_y_vel = Enemies[i].enemy_y_vel_float +
                                     (SIN_LOOK[angle_idx] * fltEnemyFiringRate[0]);

                    /* Use enemy type + 4 as flare index */
                    int flare_idx = Enemies[i].enemy_type + 4;

                    AddParticleByVector(
                        2,                              /* detect_collisions = 2 for enemy bullets */
                        LARGE_PARTICLE,                 /* img_size_type */
                        flare_idx,                      /* img_index */
                        Enemies[i].enemy_x_float,       /* x */
                        Enemies[i].enemy_y_float,       /* y */
                        fire_x_vel,                     /* x_vel */
                        fire_y_vel,                     /* y_vel */
                        80,                             /* max_life */
                        10                              /* damage */
                    );
                }
            }
        }
    }
}

/*
 * Create ship explosion effect - mirrors assembly _ShipExplode (ai.asm:718-927)
 * The assembly uses different particle configurations based on size and hit parameters:
 * - Small explosions (size=0, hit=0): 20 loops x 3 particles = 60 particles, large flares, no damage
 * - Big explosions (size=1, hit=0): 100 loops x 6 particles = 600 particles, mixed flares, damage=100
 * - Hit effects (hit=1): 3-5 particles, smoke puffs, no damage
 */
void ShipExplode(float x, float y, int large, int damage) {
    extern float fltPSpeed1, fltPSpeed2, fltPSpeed3, fltPSpeed4;
    float fltTemp3 = 5.3f;

    if (damage == 1) {
        /* Hit effect - creates smoke puffs - mirrors assembly lines 888-927 */
        uint8_t angle = Rand() % 256;

        AddParticle(0, SMALL_PARTICLE, 14, x, y, angle, fltTemp3, 15, 0);
        AddParticle(0, LARGE_PARTICLE, 2, x, y, angle + 85, fltTemp3, 15, 0);
        AddParticle(0, LARGE_PARTICLE, 3, x, y, angle + 170, fltTemp3, 15, 0);

        if (large == 10) {
            AddParticle(0, SMALL_PARTICLE, 14, x, y, angle + 193, fltTemp3, 15, 0);
            AddParticle(0, SMALL_PARTICLE, 14, x, y, angle + 261, fltTemp3, 15, 0);
        }
    } else if (large == 1) {
        /* Big explosion - 100 loops, 6 particles each - mirrors assembly lines 785-886 */
        for (int i = 0; i < 100; i++) {
            AddParticle(1, SMALL_PARTICLE, 4, x, y, Rand() % 256, fltPSpeed1, 200, 100);
            AddParticle(1, SMALL_PARTICLE, 4, x, y, Rand() % 256, fltPSpeed3, 200, 100);
            AddParticle(1, LARGE_PARTICLE, 5, x, y, Rand() % 256, fltPSpeed2, 210, 100);
            AddParticle(1, SMALL_PARTICLE, 5, x, y, Rand() % 256, fltPSpeed1, 220, 100);
            AddParticle(1, SMALL_PARTICLE, 5, x, y, Rand() % 256, fltPSpeed3, 220, 100);
            AddParticle(1, SMALL_PARTICLE, 6, x, y, Rand() % 256, fltPSpeed4, 230, 100);
        }
    } else {
        /* Small explosion - 20 loops, 3 particles each - mirrors assembly lines 737-783 */
        for (int i = 0; i < 20; i++) {
            AddParticle(0, LARGE_PARTICLE, 3, x, y, Rand() % 256, fltPSpeed1, 10, 0);
            AddParticle(0, LARGE_PARTICLE, 7, x, y, Rand() % 256, fltPSpeed2, 20, 0);
            AddParticle(0, LARGE_PARTICLE, 8, x, y, Rand() % 256, fltPSpeed3, 30, 0);
        }
    }
}

/*
 * Drop a nuke - massive circular shockwave - mirrors assembly _DropNuke (ai.asm:675-710)
 * Creates 4 particles per angle increment in a circular pattern
 */
void DropNuke(float x, float y) {
    /* Loop through angles 0-255 in increments of 5 - mirrors assembly lines 683-708 */
    for (int angle = 0; angle < 256; angle += 5) {
        /* Create 4 particles at different angles and speeds - mirrors assembly lines 686-703 */
        AddParticle(1, LARGE_PARTICLE, 5, x, y, (uint8_t)angle, fltEnemyFiringRate[4], 100, 20);
        AddParticle(1, SMALL_PARTICLE, 15, x, y, (uint8_t)(angle + 1), fltEnemyFiringRate[2], 100, 20);
        AddParticle(1, LARGE_PARTICLE, 5, x, y, (uint8_t)(angle + 2), fltEnemyFiringRate[1], 100, 20);
        AddParticle(1, SMALL_PARTICLE, 15, x, y, (uint8_t)(angle + 3), fltEnemyFiringRate[0], 100, 20);
    }
}
